﻿// Copyright (c) Microsoft Corporation.  All rights reserved.

function TestRunGraph(kpi, viewDiv, legendDiv, testRunMonitor, framework) {

    var refreshTimer = null;
    var instanceIdTimer = null;
    var plot = null;
    var plotData = [];
    var computedValues = [];
    var waitingForData = false;
    var allSamplesReceived = false;
    var sampleRate = 0;
    var dateTimeEncoding = framework.GetWebConfiguration().DateTimeEncoding;
    var visible = true;
    var pendingCounters = 0;
    var counterColors = null;
    var testRunDuration = 0;
    var currentDuration = 0;
    var rebuildGraph = false;
    var redrawGraph = false;
    var applicationTabName = "Application";
    var isAutKpi = (kpi["GroupName"] === applicationTabName);
    var autKpiFetchedTimeout = false;
    var waitingForAutCounters = false;
    var AutKpiFetched = false;
    var instanceIdTimerForAutKpi = null;
    var refreshPlotFunction;
    var batchSize = 500;
    var stringsResource = framework.GetResource().GetString;
    var Count = 1;
    var applicationGroupName = "Application";

    //TODO: Initialize this at run time.
    var colors = {
        'User Load': '#AD73C9', 'Errors/Sec': '#D80010', 'Tests/Sec': '#579E57', 'Pages/Sec': '#025893', 'Requests/Sec': '#F2700F',
        'Avg. Test Time': '#579E57', 'Avg. Page Time': '#025893', 'Avg. Response Time': '#F2700F'
    };

    var allCounterInstancesFound = false;
    var waitingForCounterInstances = false;
    var remainingCounterInstancesAttempts = 30;
    var remainingCounterInstancesAttemptsAut = 30;
    var remainingCounterInstancesAttemptsToFetchAutCounters = 60;
    var remainingRefreshPlotAttemptForCompletedRuns = 10

    $(window).resize(resizeGraph);

    function resizeGraph() {
        if (plot != null && visible && viewDiv.width() > 0) {
            buildandShowChart();
        }
    }

    function stopTimer() {
        if (refreshTimer != null) {
            window.clearInterval(refreshTimer);
            refreshTimer = null;
        }
        if (instanceIdTimer != null) {
            window.clearInterval(instanceIdTimer);
            instanceIdTimer = null;
        }
        if (instanceIdTimerForAutKpi != null) {
            window.clearInterval(instanceIdTimerForAutKpi);
            instanceIdTimerForAutKpi = null;
        }
    }

    function isCounterInstanceAvailable(counterInstanceId) {
        return (!framework.IsNullOrUndefined(counterInstanceId) && counterInstanceId.length !== 0);
    }

    function isCounterTypeNumberOfItems(counterType) {
        return (counterType == "NumberOfItems32") || (counterType == "NumberOfItems64"); /*PerformanceCounterType.NumberOfItems32*/
    }

    function isCounterTypeAverageTimer32(counterType) {
        return (counterType == "AverageTimer32"); /*PerformanceCounterType.AverageTimer32*/
    }

    function isCounterTypeRateOfCountsPerSecond64(counterType) {
        return (counterType == "RateOfCountsPerSecond64"); /*PerformanceCounterType.RateOfCountsPerSecond64*/
    }

    function isCounterTypeTimer100NsInverse(counterType) {
        return (counterType == "Timer100NsInverse"); /*PerformanceCounterType.RateOfCountsPerSecond64*/
    }

    function computeSamplesAverage(currentSample, previousSample) {
        //NOTE: Original Source - $/DevDiv/D11Rel/Q11W/vset/QTools/LoadTest/WebLoadTest/Engine/Results/LoadTestResult/DataSeries.cs

        if (isCounterTypeNumberOfItems(currentSample.CounterType)) {
            return currentSample.ComputedValue;
        }

        if (isCounterTypeRateOfCountsPerSecond64(currentSample.CounterType)) {
            if (previousSample == null) {
                return currentSample.ComputedValue;
            }

            if (previousSample.ComputedValue == 0 && currentSample.TimeStamp == previousSample.TimeStamp) {
                return "NaN";
            }

            return (currentSample.RawValue - previousSample.RawValue) /
                    ((currentSample.TimeStamp - previousSample.TimeStamp) / currentSample.SystemFrequency);
        }

        if (isCounterTypeAverageTimer32(currentSample.CounterType)) {
            if (previousSample == null) {
                return currentSample.ComputedValue;
            }

            if (previousSample.ComputedValue == 0 && currentSample.BaseValue == previousSample.BaseValue) {
                return "NaN";
            }

            return ((currentSample.RawValue - previousSample.RawValue) / previousSample.CounterFrequency)
					/ (currentSample.BaseValue - previousSample.BaseValue);
        }

        if (isCounterTypeTimer100NsInverse(currentSample.CounterType)) {
            if (previousSample == null) {
                return currentSample.ComputedValue;
            }
            else {
                return (currentSample.ComputedValue + previousSample.ComputedValue) / 2.0;
            }
        }

        return "NaN";
    }

    function computeValueFromSamples(currentSample, previousSample) {
        if (isCounterTypeRateOfCountsPerSecond64(currentSample.CounterType)) {
            if (currentSample.ComputedValue == 0 && currentSample.TimeStamp == previousSample.TimeStamp) {
                return "NaN";
            }
            return currentSample.ComputedValue;
        }

        if (isCounterTypeAverageTimer32(currentSample.CounterType)) {
            if (currentSample.ComputedValue == 0 && currentSample.BaseValue == previousSample.BaseValue) {
                return "NaN";
            }
            return currentSample.ComputedValue;
        }

        return currentSample.ComputedValue;
    }

    function counterSampleReceivedHelper(data) {
        var nextFetch = false;
        if (data) {
            $.each(kpi.CounterInstances, function (counterInstanceIndex, counterInstance) {
                var counterSamples;
                if (isCounterInstanceAvailable(counterInstance.CounterInstanceId)) {

                    var allMatchingCounterInstanceSamples = $.grep(data.Values, function (counterInstanceSample, counterInstanceSampleIndex) {
                        return counterInstanceSample.CounterInstanceId === counterInstance.CounterInstanceId;
                    });

                    var allMatchingCounterSamples = [];
                    $.each(allMatchingCounterInstanceSamples, function (counterInstanceSampleIndex, counterInstanceSample) {
                        $.each(counterInstanceSample.Values, function (counterSampleIndex, counterSample) {
                            allMatchingCounterSamples.push(counterSample);
                        });
                    });

                    var computedData = computedValues[counterInstanceIndex];
                    if (isAutKpi) {
                        if (computedData.lastTimeValue != null) {
                            counterSamples = $.grep(allMatchingCounterSamples, function (counterSample, counterSampleIndex) {
                                return Date.parse(counterSample.IntervalEndDate) > Date.parse(computedData.lastTimeValue);
                            });
                        }
                        else {
                            counterSamples = allMatchingCounterSamples;
                        }
                    }
                    else {
                        counterSamples = $.grep(allMatchingCounterSamples, function (counterSample, counterSampleIndex) {
                            return counterSample.IntervalNumber > computedData.lastInterval;
                        });
                    }
                  
                    counterSampleReceived(counterInstanceIndex, counterSamples);
                    counterInstanceProcessed();
                }
            });
	    if (data.TotalSamplesCount >= data.MaxBatchSize) {
                nextFetch = true;
            }
        }
        return nextFetch;
    }

    function counterSampleReceived(counterIndex, counterSamples) {
        if (plot != null && counterSamples.length > 0) {
            var element = plotData[counterIndex];
            var computedData = computedValues[counterIndex];
            var newValues = [];
            var newIntervals = [];
            var newTimeValues = [];
            var oldMultiplier = computedData.multiplier;

            computedData.maxValue = 0;
            computedData.minValue = -1;
            computedData.multiplier = 1;
            computedData.lastValue = 0;
            computedData.range = 100;

            $.each(counterSamples, function (index, counterSample) {
                var sampleTime = (new DateTime(counterSample.IntervalEndDate, dateTimeEncoding)).GetJSDate();

                if (counterSample.IntervalNumber != 0) {
                    if (counterSample.IntervalNumber == 1) {
                        counterSampleReceived.firstCounterCount = ++counterSampleReceived.firstCounterCount || 1;
                        if (counterSampleReceived.firstCounterCount == 1) {
                            elsFramework.GetCodeMarkers().fire(elsFramework.GetCodeMarkers().CodeMarkerValues.PerfELS_GetFirstGraphCounterEnd);
                        }
                    }
                    var value = computeValueFromSamples(counterSample, computedData.prevSample);
                    if (value != "NaN") {
                        computedData.originalSampleValue.push(value);
                        newValues.push(value);
                        computedData.intervalNumbers.push(counterSample.IntervalNumber);
                        newIntervals.push(counterSample.IntervalNumber);
                        computedData.timeValues.push(sampleTime);
                        newTimeValues.push(sampleTime);
                    }

                    if (currentDuration != testRunDuration) {
                        if (isAutKpi) {
                            //In service side, if ExecutionStartedDate and CreatedDate are same,we only fill up the CreatedDate field of TestRun object.
                            var currentTest = testRunMonitor.GetTestRun();
                            var executionStarted = (currentTest.ExecutionStartedDate == null) ? currentTest.CreatedDate : currentTest.ExecutionStartedDate;
                            var timeDiff = (sampleTime.getTime() - ((new DateTime(executionStarted, dateTimeEncoding)).GetJSDate()).getTime()) / 1000;
                        }
                        else {
                            var timeDiff = (sampleTime.getTime() - computedData.firstSampleTime.getTime()) / 1000;
                        }
                        if (timeDiff > (currentDuration - 60)) {
                            if ((currentDuration * 2) < testRunDuration) {
                                currentDuration *= 2;
                            } else {
                                currentDuration = testRunDuration;
                            }
                            rebuildGraph = true;
                        }
                    }
                } else {
                    computedData.firstSampleTime = (new DateTime(counterSample.IntervalEndDate, dateTimeEncoding)).GetJSDate();
                    computedData.firstSample = counterSample;
                }
                computedData.lastInterval = counterSample.IntervalNumber;
                computedData.prevSample = counterSample;
                computedData.lastTimeValue = counterSample.IntervalEndDate;
            });

            var totalValidSamples = 0;
            var sampleSum = 0;

            $.each(computedData.originalSampleValue, function (index, sampleValue) {
                if (computedData.minValue == -1 || computedData.minValue > sampleValue) {
                    computedData.minValue = sampleValue;
                }

                if (computedData.maxValue < sampleValue) {
                    computedData.maxValue = sampleValue;
                }

                computedData.lastValue = sampleValue;

                sampleSum += sampleValue;
                totalValidSamples++;
            });

            if (totalValidSamples > 0) {
                if (isAutKpi) {
                    computedData.average = sampleSum / totalValidSamples;
                }
                else {
                    computedData.average = computeSamplesAverage(computedData.prevSample, computedData.firstSample);
                }
            }

            var maxValue = computedData.maxValue;

            while (maxValue != 0 && maxValue <= 10) {
                computedData.multiplier *= 10;
                maxValue *= 10;
                computedData.range /= 10;
            }

            while (maxValue != 0 && maxValue > 100) {
                computedData.multiplier /= 10;
                maxValue /= 10;
                computedData.range *= 10;
            }

            if (computedData.originalSampleValue.length > 1 && element.marker.enabled) {
                element.marker.enabled = false;
                rebuildGraph = true;
            }

            if (oldMultiplier !== computedData.multiplier) {
                element.data = [];
                if (isAutKpi) {
                    //In service side, if ExecutionStartedDate and CreatedDate are same,we only fill up the CreatedDate field of TestRun object.
                    var currentTest = testRunMonitor.GetTestRun();
                    var executionStarted = (currentTest.ExecutionStartedDate == null) ? currentTest.CreatedDate : currentTest.ExecutionStartedDate;
                    var executionStartedTime = (new DateTime(executionStarted, dateTimeEncoding)).GetJSDate();
                    $.each(computedData.originalSampleValue, function (index, sampleValue) {
                        element.data.push([computedData.timeValues[index] - executionStartedTime, sampleValue * computedData.multiplier]);
                    });
                }
                else {
                    $.each(computedData.originalSampleValue, function (index, sampleValue) {
                        element.data.push([computedData.intervalNumbers[index] * sampleRate * 1000, sampleValue * computedData.multiplier]);
                    });
                }
                if (plot && plot.series)
                    plot.series[counterIndex].setData(element.data, true);
            } else {
                $.each(newValues, function (index, sampleValue) {
                    if (plot && plot.series) {
                        if (isAutKpi) {
                            //In service side, if ExecutionStartedDate and CreatedDate are same,we only fill up the CreatedDate field of TestRun object.
                            var currentTest = testRunMonitor.GetTestRun();
                            var executionStartedDate = (currentTest.ExecutionStartedDate == null) ? currentTest.CreatedDate : currentTest.ExecutionStartedDate;
                            plot.series[counterIndex].addPoint([newTimeValues[index] - (new DateTime(executionStartedDate, dateTimeEncoding)).GetJSDate(), sampleValue * computedData.multiplier], false);
                        }
                        else {
                            plot.series[counterIndex].addPoint([newIntervals[index] * sampleRate * 1000, sampleValue * computedData.multiplier], false);
                        }
                    }
                });
                redrawGraph = true;
                newValues = [];
                newIntervals = [];
            }
        }
    }

    function updateLegends() {
        var tableId = "counterTable";
        var tableDiv = createCounterTable(tableId);
        $.each(computedValues, function (index, rowData) {
            var isChecked = plotData[index].visible ? 'checked' : '';
            var rowId = "check" + index;
            var checkbox = "<td class='checkBoxCell'><input class='checkBoxClass' id='" + rowId + "' type='checkbox' " + isChecked + "/></td>";
            var tableContents = "<tr>";
            tableContents += "<td class='firstColumn'><table><tr>" + checkbox + "<td class='counterBoxCell'><div class='counterBox'></div></td><td class='counterText'>" + rowData.counterName + "</td></tr></table></td><td>" + rowData.counterUnits + "</td>";
            if (rowData.minValue != -1) {
                tableContents += "<td>" + rowData.range + "</td>";
                tableContents += "<td>" + roundValueForTable(rowData.minValue) + "</td>";
                tableContents += "<td>" + roundValueForTable(rowData.maxValue) + "</td>";
                tableContents += "<td>" + roundValueForTable(rowData.average) + "</td>";
                tableContents += "<td>" + roundValueForTable(rowData.lastValue) + "</td>";
            } else {
                tableContents += "<td>-</td><td>-</td><td>-</td><td>-</td><td>-</td>";
            }
            tableContents += "</tr>";
            tableDiv.append(tableContents);
            if (counterColors != null) {
                var lastRow = tableDiv.find('.counterBox:last');
                lastRow.html("<svg class='svgCounterCell'><rect x='0' y='0' width='13px' height='13px' fill='" + counterColors[index] + "' /></svg>");
            }
            rowId = '#' + rowId;
            $(rowId).click(legendCheckBoxClicked);
        });
        SelectAllLegendCheckBoxes();

    }

    /* Functions for SelectAll Functionality in CheckBox */
    function legendCheckBoxClicked() {
        var rowId = $(this).attr('id');
        var id = parseInt(rowId.substring(5));
        var series = plot.series[id];

        var isChecked = $(this).is(':checked');

        series.setVisible(isChecked, true);
        plotData[id].visible = isChecked;

        SelectAllLegendCheckBoxes();
    }

    function SelectAllLegendCheckBoxes() {
        if ($(".checkBoxClass:checkbox").length == $(".checkBoxClass:checkbox:checked").length) {
            $("#selectAllCheckBox").attr('checked', true);
        }
        else {
            $("#selectAllCheckBox").attr('checked', false);
        }
    }

    function SelectAllLegendBoxClicked() {
        var isSelectAllBoxSelected = $(this).is(':checked');
        $(".checkBoxClass:checkbox").each(function () {
            if (isSelectAllBoxSelected != $(this).is(':checked')) {
                this.click();
            }
        });
    }
    /* End of SelectAll Functions */

    function createCounterTable(tableId) {
        var tableHtml = "<table id='" + tableId + "'><tr>";
        var checkbox = "<input id='selectAllCheckBox' type='checkbox' />";
        tableHtml += "<th class='firstColumn'>" + checkbox + stringsResource("GraphColumnCounter") + "</th>";
        tableHtml += getTableColumnHeading(stringsResource("GraphColumnUnits"));
        tableHtml += getTableColumnHeading(stringsResource("GraphColumnRange"));
        tableHtml += getTableColumnHeading(stringsResource("GraphColumnMin"));
        tableHtml += getTableColumnHeading(stringsResource("GraphColumnMax"));
        tableHtml += getTableColumnHeading(stringsResource("GraphColumnAverage"));
        tableHtml += getTableColumnHeading(stringsResource("GraphColumnLast"), true);
        tableHtml += "</tr></table>";
        legendDiv.html(tableHtml);
        tableId = "#" + tableId;
        $("#selectAllCheckBox").click(SelectAllLegendBoxClicked);

        return $(tableId);
    }

    function getTableColumnHeading(heading, isLast) {
        if (isLast) {
            return "<th class='lastColumn'>" + heading + " </th>";
        }
        else {
            return "<th>" + heading + " </th>";
        }
    }

    //rounds to 2 decimal places
    function roundValueForTable(originalValue) {
        var newValue = Math.round(originalValue * 100) / 100;
        return newValue;
    }

    function refreshPlot() {

        if (waitingForData) {
            return;
        }

        var testRun = testRunMonitor.GetTestRun();

        if (isAutKpi && testRun.ExecutionStartedDate === testRun.CreatedDate) {
            return;
        }

        if (isTestRunComplete())
        {
            if(remainingRefreshPlotAttemptForCompletedRuns == 0)
            {
                return;
            }
            else
            {
                remainingRefreshPlotAttemptForCompletedRuns--;
            }
        }
        var testServiceClient = framework.GetTestServiceClient();
        var counterSampleQueryDetails = [];
        var fromInterval = -1;
        var toInterval = -1;

        $.each(kpi.CounterInstances, function (counterInstanceIndex, counterInstance) {
            if (isCounterInstanceAvailable(counterInstance.CounterInstanceId)) {

                fromInterval = 0;
                toInterval = -1;

                if (!isAutKpi) {
                    fromInterval = (computedValues[counterInstanceIndex].lastInterval + 1) * sampleRate;
                    toInterval = fromInterval + (batchSize * sampleRate);
                }
                else if (computedValues[counterInstanceIndex].lastTimeValue) {
                    fromInterval = (((new DateTime(computedValues[counterInstanceIndex].lastTimeValue, dateTimeEncoding)).GetJSDate()).getTime() - ((new DateTime(testRunMonitor.GetTestRun().ExecutionStartedDate, dateTimeEncoding)).GetJSDate()).getTime()) / 1000;
                    fromInterval += 1;
                }

                //avoid sample query where fromInterval exceeds run duration
                //as that wont return any data
                if (fromInterval <= testRunDuration) {
                    counterSampleQueryDetails.push({ CounterInstanceId: counterInstance.CounterInstanceId, FromInterval: fromInterval, ToInterval: toInterval });
                }
            }
        });

        if (counterSampleQueryDetails.length <= 0) {
            return;
        }

        waitingForData = true;

        //Fetch data in batches of 500
        testServiceClient.GetTestRunCounterSamples(testRun, counterSampleQueryDetails, function (data) {
            if (!framework.IsNullOrUndefined(data) && data.Count > 0) {
                var fetchNext = counterSampleReceivedHelper(data);
                if (fetchNext == true) {
                    refreshPlotFunction();
                }
                else {
                    if (isTestRunComplete()) {
                        allSamplesReceived = true;
                        stopTimer();
                    }
                }

                if (isAutKpi && !isTestRunComplete()) {

                    // Update the Refresh Rate 
                    var refreshTime;

                    $.each(data.Values, function (counterInstanceSampleIndex, counterInstanceSample) {
                        if (refreshTime) {
                            if (counterInstanceSample.NextRefreshTime < refreshTime) {
                                refreshTime = counterInstanceSample.NextRefreshTime;
                            }
                        } else {
                            refreshTime = counterInstanceSample.NextRefreshTime;
                        }
                    });

                    if (refreshTime) {
                        var nextRefreshTime = (new DateTime(refreshTime, dateTimeEncoding)).GetJSDate();
                        var timeDiff = nextRefreshTime.getTime() - (new Date()).getTime();
                        if (timeDiff < 0) {
                            timeDiff = sampleRate * 1000;
                        }
                        if (refreshTimer != null) {
                            window.clearInterval(refreshTimer);
                        }
                        refreshTimer = window.setInterval(refreshPlotFunction, timeDiff);
                    }
                }
            } else {
                pendingCounters = 0;
                waitingForData = false;
                counterInstanceProcessed();
            }
        },
        function (error) {
            framework.LogError("Get samples error: " + error);
            pendingCounters = 0;
            counterInstanceProcessed();
        });
    }

    function counterInstanceProcessed() {
        pendingCounters--;

        if (pendingCounters <= 0) {
            waitingForData = false;

            if (visible) {
                if (rebuildGraph) {
                    resizeGraph();
                    rebuildGraph = false;
                }
                if (redrawGraph) {
                    plot.redraw();
                    redrawGraph = false;
                }
                updateLegends();
            }
        }
    }

    function refreshCounterInstances() {
        if (!isAutKpi) {
            refreshCounterInstancesGeneral();
        }
        else {
            refreshCounterInstancesAut();
        }

    }

    function refreshCounterInstancesGeneral() {
        if (!allCounterInstancesFound && !waitingForCounterInstances && remainingCounterInstancesAttempts > 0) {

            waitingForCounterInstances = true;
            remainingCounterInstancesAttempts--;

            framework.GetTestServiceClient().GetKeyPerformanceIndicators(testRunMonitor.GetTestRun(), testRunMonitor.GetKpiGroupNames(), function (kpis) {
                waitingForCounterInstances = false;
                allCounterInstancesFound = true;
                refreshcounterInstancesHelper(kpis);
            },
            function (error) {
                framework.LogError("Get KPIs error: " + error);
                waitingForCounterInstances = false;
            }
            );
        }
    }

    function refreshCounterInstancesAut() {
        if (!allCounterInstancesFound && !waitingForCounterInstances && remainingCounterInstancesAttemptsAut > 0) {

            waitingForCounterInstances = true;
            remainingCounterInstancesAttemptsAut--;

            framework.GetTestServiceClient().GetKeyPerformanceIndicators(testRunMonitor.GetTestRun(), testRunMonitor.GetKpiGroupNames(), function (kpis) {
                waitingForCounterInstances = false;
                allCounterInstancesFound = true;
                refreshcounterInstancesHelper(kpis);
            },
            function (error) {
                framework.LogError("Get Aut KPIs error: " + error);
                waitingForCounterInstances = false;
            }
            );
        }
    }

    function refreshcounterInstancesHelper(kpis) {
        var newCountersReceived = false;
        if (kpis) {
            $.each(kpis, function (kpiGroupIndex, kpiGroup) {
                if (kpi.GroupName != kpiGroup.GroupName) { return; }
                $.each(kpi.CounterInstances, function (counterInstanceIndex, reqdCounterInstance) {
                    if (isCounterInstanceAvailable(reqdCounterInstance.CounterInstanceId)) {
                        return;
                    }

                    $.each(kpiGroup.CounterInstances, function (counterInstanceIndex2, availableCounterInstance) {
                        if (counterInstancesMatch(reqdCounterInstance, availableCounterInstance)) {
                            newCountersReceived = true;
                            reqdCounterInstance.CounterInstanceId = availableCounterInstance.CounterInstanceId;
                            reqdCounterInstance.MachineName = availableCounterInstance.MachineName;
                        }
                        else {
                            allCounterInstancesFound = false;
                        }
                    });
                });
            });
        }

        if (newCountersReceived) {
            refreshPlotFunction();
        }

        if (allCounterInstancesFound) {
            if (instanceIdTimer != null) {
                window.clearInterval(instanceIdTimer);
                instanceIdTimer = null;
            }
        }
    }

    function counterInstancesMatch(reqdCounterInstance, availableCounterInstance) {
        return reqdCounterInstance.CounterName == availableCounterInstance.CounterName
            && reqdCounterInstance.InstanceName == availableCounterInstance.InstanceName;
    }

    function getTickSize(duration) {
        var requiredTickSize = 5;

        while ((duration / (requiredTickSize)) > 12) {
            requiredTickSize += 5;
        }

        return requiredTickSize;
    }

    function getTimeSpanString(val) {
        var duration = new Date(val);
        var hours = (duration.getUTCHours() + (duration.getUTCDate() - 1) * 24);
        if (hours < 10) {
            hours = "0" + hours;
        }
        var minutes = duration.getUTCMinutes();
        if (minutes < 10) {
            minutes = "0" + minutes;
        }
        var seconds = duration.getUTCSeconds();
        if (seconds < 10) {
            seconds = "0" + seconds;
        }

        var formattedString = "";
        if (hours != "00") {
            formattedString += hours + ":";
        }

        formattedString += minutes + ":" + seconds;

        return formattedString;
    }

    function ticksFormatterX() {
        return getTimeSpanString(this.value);
    }

    //Returns colors selected by plot for different counters
    function getPlotColors() {
        var colorArray = new Array();
        var series = plot.series;
        for (var i = 0; i < series.length; ++i) {
            colorArray.push(series[i].color);
        }
        return colorArray;
    }

    function getGraphOptions() {
        if (plotData.length == 0) {
            $.each(kpi.CounterInstances, function (counterInstanceIndex, counterInstance) {
                var isVisible = (isAutKpi) ? counterInstance.IsPreselectedCounter : true;
                plotData.push({ name: counterInstance.CounterName, data: [], visible: isVisible, color: colors[counterInstance.CounterName], marker: { enabled: true, radius: 2, symbol: 'circle' } });

                var computedValue = computedValues[counterInstanceIndex];

                if (!computedValue) {
                    computedValues.push({
                        counterName: counterInstance.CounterName,
                        counterUnits: counterInstance.CounterUnits,
                        minValue: -1,
                        maxValue: 0,
                        multiplier: 1,
                        range: 100,
                        average: 0,
                        lastValue: 0,
                        lastInterval: -1,
                        lastTimeValue: null,
                        originalSampleValue: [],
                        intervalNumbers: [],
                        timeValues: [],
                        firstSample: null,
                        prevSample: null,
                        firstSampleTime: null
                    });
                }
            });
        }

        return {
            chart: { type: 'line', renderTo: viewDiv.attr('id'), backgroundColor: null, marginRight: 25, animation: false },
            series: plotData,
            plotOptions: {
                series: { shadow: false, marker: { enabled: true, radius: 2, symbol: 'circle' }, stickyTracking: false, animation: false, connectNulls: true },
                line: { point: { events: { mouseOut: function () { this.series.chart.tooltip.hide(); } } } }
            },
            yAxis: {
                tickInterval: 20, min: 0, max: 100, gridLineWidth: 1, title: { text: null }, labels: { useHTML: true },
                gridLineColor: Plugin.Theme.getValue("plugin-textbox-border-color"),
                lineWidth: 0
            },
            xAxis: {
                type: 'datetime',
                tickInterval: getTickSize(currentDuration) * 1000,
                labels: { formatter: ticksFormatterX },
                min: 0,
                max: currentDuration * 1000,
                gridLineWidth: 1,
                lineWidth: 0,
                gridLineColor: Plugin.Theme.getValue("plugin-textbox-border-color")
            },
            legend: {
                enabled: false
            },
            tooltip: {
                animation: false,
                borderWidth: 0,
                borderRadius: 0,
                backgroundColor: "none",
                shadow: false,
                shared: false,
                hideDelay: 0,
                formatter: function () {
                    return "<div class='highcharts-tooltip-div'>" + this.series.name
                            + "<br\>" + stringsResource("GraphTime", getTimeSpanString(this.x))
                            + "<br\>" + stringsResource("GraphValue", (this.y / computedValues[this.series.index].multiplier).toFixed(4)) + "</div>";
                },
                useHTML: true
            },
            credits: { enabled: false },
            title: { text: null },
            colors: ['#8DC54B', '#F58B1F', '#7F1725', '#009BCC', '#68217A', '#077ACC', '#F2700F', '#147A7C', '#AE3CBA', '#00188F', '#9DBFE9', '#804097', '#525151', '#EC008C', '#339947', '#9FB1DB', '#748188', '#F9CCE8', '#DB552C', '#C3D84C', '#FAEF67', '#005F31', '#E31E26', '#FFCC05']
        };
    }

    function setUnits() {
        var seconds = [];
        for (var i = 1; i <= currentDuration; i++)
            seconds.push(i);
        Highcharts.Axis.prototype.defaultOptions.units = [[
            'millisecond',
            [1, 2, 5, 10, 20, 25, 30, 40, 50, 100, 200, 500]
        ], [
            'second',
            seconds
        ], [
            'minute',
            [1, 2, 3, 4, 5, 6, 7, 8, 9, 10,
             11, 12, 13, 14, 15, 16, 17, 18, 19, 20,
             21, 22, 23, 24, 25, 26, 27, 28, 29, 30,
             31, 32, 33, 34, 35, 36, 37, 38, 39, 40,
             41, 42, 43, 44, 45, 46, 47, 48, 49, 50,
             51, 52, 53, 54, 55, 56, 57, 58, 59]
        ], [
            'hour',
            [1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12]
        ]];
    }

    function buildandShowChart() {
        viewDiv.empty();

        setUnits();
        plot = new Highcharts.Chart(getGraphOptions());

        counterColors = getPlotColors();
        updateLegends();

        plot.redraw();
        buildandShowChart.count = ++buildandShowChart.count || 1;
        elsFramework.GetCodeMarkers().fire(elsFramework.GetCodeMarkers().CodeMarkerValues.PerfELS_OpenTabEnd);
    }

    function drawEmptyGraph() {
        viewDiv.empty();

        setUnits();
        var options = getGraphOptions();
        var series = { data: [] };
        options.series.push(series);
        plot = new Highcharts.Chart(options);

        counterColors = getPlotColors();
        updateLegends();

        plot.redraw();
    }

    function fetchAutCounters() {
        if (waitingForAutCounters || AutKpiFetched) {
            return;
        }

        if (remainingCounterInstancesAttemptsToFetchAutCounters < 0) {
            $("#waitingInfoMessage").html();
            $("#waitingInfo").hide();
            $("#MoreInfo").html("");
            autKpiFetchedTimeout = true;
            return;
        }

        waitingForAutCounters = true;
        remainingCounterInstancesAttemptsToFetchAutCounters--;
        var testRun = testRunMonitor.GetTestRun();
        framework.GetTestServiceClient().GetKeyPerformanceIndicators(testRun, testRunMonitor.GetKpiGroupNames(), function (kpiGroups) {

            if (kpiGroups != null && kpiGroups.length > 0) {

                $.each(kpiGroups, function (kpiGroupIndex, kpiGroup) {
                    if (kpi.GroupName != kpiGroup.GroupName) { return; }

                    if (kpiGroup && kpiGroup.CounterInstances && kpiGroup.CounterInstances.length > 0) {
                        plotData = [];
                        AutKpiFetched = true;

                        var newCounterInstances = $.grep(kpiGroup.CounterInstances, function (counterInstance, counterInstanceIndex) {
                            var isNew = true;
                            $.each(kpi.CounterInstances, function (existingCounterInstanceIndex, existingCounterInstance) {

                                if (counterInstancesMatch(existingCounterInstance, counterInstance)) {
                                    isNew = false;
                                }
                            });

                            return isNew;
                        });

                        $.each(newCounterInstances, function (newCounterInstanceIndex, newCounterInstance) {
                            kpi.CounterInstances.push(newCounterInstance);
                        });
                    }
                });

                if (instanceIdTimerForAutKpi != null) {
                    window.clearInterval(instanceIdTimerForAutKpi);
                    instanceIdTimerForAutKpi = null;
                }
                $("#waitingInfoMessage").html();
                $("#waitingInfo").hide();
                var autInfoId = "#kpiInfo_" + kpi.GroupName;
                // To Check : If Some Error happens while we are requesting for Application Insights Counters. Even if we get the Counters Show Error message
                if (!$(autInfoId).hasClass('kpiInfoShow')) {
                    showGeneralInfo("AutDelayMessage");
                }
                $("#MoreInfo").html("");
                fetchAutCounters.showGraphCount = ++fetchAutCounters.showGraphCount || 1;
                if (fetchAutCounters.showGraphCount == 1) {
                    elsFramework.GetCodeMarkers().fire(elsFramework.GetCodeMarkers().CodeMarkerValues.PerfELS_GetFirstGraphCounterOnApplicationTabEnd);
                }
                showCurrentGraph();
            }
            waitingForAutCounters = false;
        }, function (error) {
            framework.LogError("Get Aut KPIs error: " + error);
            waitingForAutCounters = false;
        });
    }

    function showCurrentGraph(isSwitchKpi) {
        isSwitchKpi = (framework.IsNullOrUndefined(isSwitchKpi)) ? false : isSwitchKpi;
        if (currentDuration > 0) {
            buildandShowChart();
            if (isSwitchKpi === true) {
                elsFramework.GetCodeMarkers().fire(elsFramework.GetCodeMarkers().CodeMarkerValues.PerfELS_ShowGraphOnFirstSwitchEnd);
            }
            if (!allSamplesReceived && refreshTimer == null) {
                if (isSwitchKpi === true) {
                    elsFramework.GetCodeMarkers().fire(elsFramework.GetCodeMarkers().CodeMarkerValues.PerfELS_ShowNextSampleOnSwitchEnd);
                }
                refreshCounterInstances();
                instanceIdTimer = window.setInterval(refreshCounterInstances, 2 * 1000);
                refreshPlotFunction();
                refreshTimer = window.setInterval(refreshPlotFunction, sampleRate * 1000);
                if (isSwitchKpi === true) {
                    elsFramework.GetCodeMarkers().fire(elsFramework.GetCodeMarkers().CodeMarkerValues.PerfELS_ShowAllSamplesOnSwitchEnd);
                }
            }
            if (isSwitchKpi === true) {
                isSwitchKpi = false;
            }
        }

    }

    function isTestRunComplete() {
        var testState = testRunMonitor.GetTestState();
        if (testState.IsFinished() || testState.IsAborted() || testState.IsFailed()) {
            return true;
        }
        return false;
    }

    function showGeneralInfo(messageType) {
        var message;
        $('#InfoImage').removeClass();
        $('#InfoImage').html("");
        if (isTestRunComplete() && messageType !== "AutCounterError") {
            $("#GeneralInfo").html("");
            return;
        }
        var img = $('<img id="infoImageDiv">');
        switch (messageType) {
            case "AutCounterDelayMessage":
                message = stringsResource("AutCounterDelayMessage");
                showInfoIconInInformationWindow();
                break;
            case "AutDelayMessage":
                message = stringsResource("AutDelayMessage");
                showInfoIconInInformationWindow();
                break;
            case "AutCounterError":
                message = stringsResource("AutCounterError") + " " + stringsResource("ViewLabel") + " <a id='ViewDetailsLink'> " + stringsResource("StatusMessages") + "</a> " + stringsResource("MoreDetailsLabel");
                $('#InfoImage').addClass('iconWarning');
                break;
            case "ClearDiv":
                message = "";
                break;
        }
        $("#GeneralInfo").html(message);

        if (messageType === "AutCounterError") {
            $("#GeneralInfo").delegate("#ViewDetailsLink", "click", function () {
                $('#tabs').tabs('select', 1);
                return false;
            });
        }

    }

    function showInfoIconInInformationWindow() {
        var img = $('<img id="infoImageDiv">');
        img.attr('src', Plugin.Theme.getValue('IconInfo'));
        img.attr('data-plugin-theme-src', 'IconInfo');
        img.attr('class', 'iconInfo');
        $('#InfoImage').append(img);
    }

    function showMoreInfo() {

        if (isTestRunComplete()) {
            $("#MoreInfo").html("");
            return;
        }
        var testServiceClient = framework.GetTestServiceClient();
        var currentTestRun = testRunMonitor.GetTestRun();
        var messageText = stringsResource("AutApplicationFetchText") + "<br/>";
        testServiceClient.GetTestRunCounterInstances(currentTestRun, applicationGroupName, function (autCounters) {
            var autsConfigured = getDistinctAppNames(autCounters);
            if (autsConfigured != null && autsConfigured.length > 0) {
                $.each(autsConfigured, function (index, sample) {
                    messageText = messageText + "-" + sample.Name + "<br/>";
                });
            }
            // Show Message If run duration is less than 5 minutes (5*60)
            if (currentTestRun.RunSpecificDetails.Duration < 300) {
                messageText = messageText + "<br/>" + stringsResource("AutLoadTestDurationWarning") + "<br/>";
            }

            $("#MoreInfo").html(messageText);
        });

    }

    function getDistinctAppNames(autCounters) {
        if (framework.IsNullOrUndefined(autCounters)) {
            return null;
        }
        var distinctAutNames = [];
        var autNameSet = {};
        for (var counter in autCounters) {
            if (typeof (autNameSet[autCounters[counter].MachineName]) == "undefined") {
                distinctAutNames.push(autCounters[counter].MachineName);
            }
            autNameSet[autCounters[counter].MachineName] = 0;
        }
        return distinctAutNames;
    }

    function autPreProcess() {

        /* Information/Error Div */
        $("#InformationWindow").show();
        var autInfoId = "#kpiInfo_" + kpi.GroupName;
        if ($(autInfoId).hasClass('kpiInfoShow')) {
            showGeneralInfo("AutCounterError");
        }
        /* Information/Error Div End */

        /* Navigation Bar */
        $("#navigationSection").html("<a id='AppAnalyticsUrl' >" + stringsResource('AutAppAnalyticsLink') + "</a>");
        var testServiceClient = elsFramework.GetTestServiceClient();
        testServiceClient.GetApplicationType(function (appType) {
            $("#AppAnalyticsUrl").click(function () { framework.ViewAutPortal(appType.AutPortalLink) });
            $("#navigationSection").show();
            /* Navigation Bar End */
        });

    }

    function autPostProcess() {
        /* Waiting Image Hide */
        $("#waitingInfo").hide();
        /* Application Insights URL Hide */
        $("#navigationSection").hide();
        /* Info/Error Info Hide */
        $("#InformationWindow").hide();
    }

    function showNoAutApplicationConfigured() {
        viewDiv.empty();
        legendDiv.empty();
        var message = "<span class='autErrorMessage'>" + stringsResource("AutNoApplicationConfigured") + "<br/>";
        message = message + " <a id='AutHelpLink' >" + stringsResource("LearnLabel") + "</a> " + stringsResource("AutAppApplicationHelpText") + "</span>";
        viewDiv.html(message);
        var testServiceClient = elsFramework.GetTestServiceClient();
        testServiceClient.GetApplicationType(function (appType) {
            viewDiv.delegate("#AutHelpLink", "click", function () { framework.OpenAutHelpLink(appType.ActionUriLink) });
        });

    }

    function showErrorInformation() {
        viewDiv.empty();
        legendDiv.empty();
        var message = "<span class='autErrorMessage'><span class='iconWarning'></span>" + stringsResource("AutCounterError") + " " + stringsResource("ViewLabel") + " <a id='ViewDetailsLinkOnError'> " + stringsResource("StatusMessages") + "</a> " + stringsResource("MoreDetailsLabel") + "</span>";
        viewDiv.html(message);
        viewDiv.delegate("#ViewDetailsLinkOnError", "click", function () {
            $('#tabs').tabs('select', 1);
            return false;
        });
    }

    this.isAutGraph = function () {
        return isAutKpi;
    };

    this.showErrorDiv = function () {
        showGeneralInfo("AutCounterError");
    };

    this.Show = function (isSwitchKpi) {

        isSwitchKpi = (framework.IsNullOrUndefined(isSwitchKpi)) ? false : isSwitchKpi;
        visible = true;
        var minDuration = 24 * 15;
        if (currentDuration == 0) {
            var currentTest = testRunMonitor.GetTestRun();
            testRunDuration = currentTest.RunSpecificDetails.Duration;
            sampleRate = currentTest.RunSpecificDetails.SamplingInterval;
            if (testRunDuration <= minDuration) {
                currentDuration = testRunDuration;
            }
            else {
                currentDuration = minDuration;
            }
        }

        refreshPlotFunction = refreshPlot;

        if (isAutKpi !== true) {
            showCurrentGraph(isSwitchKpi);
        }
        else {
            if (!AutKpiFetched) {

                var autInitializationErrorFlag = testRunMonitor.GetTestRun().AutInitializationError;
                if (autInitializationErrorFlag != null && autInitializationErrorFlag == true) {
                    showErrorInformation();
                    return;
                }

                var testServiceClient = elsFramework.GetTestServiceClient();
                var currentTestRun = testRunMonitor.GetTestRun();
                testServiceClient.GetTestRunCounterInstances(currentTestRun, applicationGroupName, function (autCounters) {
                    if (autCounters == null || autCounters.length == 0) {
                        /* If No Application Configured */
                        showNoAutApplicationConfigured();
                        return;
                        /* No Application Configured End    */
                    }
                    drawEmptyGraph();
                    if (Count == 1) {
                        elsFramework.GetCodeMarkers().fire(elsFramework.GetCodeMarkers().CodeMarkerValues.PerfELS_ShowCountersDefinitionsInApplicationTabEnd);
                    }
                    Count++;
                    if (!isTestRunComplete()) {
                        if (!autKpiFetchedTimeout) {
                            /* Show Initial Fetch Image */
                            $("#waitingInfoMessage").html(stringsResource("AutWaitingInfoMessage"));
                            $("#waitingInfo").show();
                            showGeneralInfo("AutCounterDelayMessage");
                            showMoreInfo();
                        }
                    }
                    /* End of Initial Fetch Image */
                    autPreProcess();

                    if (autKpiFetchedTimeout) {
                        $("#MoreInfo").html("");
                        return;
                    }

                    fetchAutCounters();
                    instanceIdTimerForAutKpi = window.setInterval(fetchAutCounters, 5 * 1000);
                });
            }
            else {
                autPreProcess();
                showCurrentGraph(isSwitchKpi);
            }
        }
    };

    this.Hide = function () {
        visible = false;
        stopTimer();
        if (isAutKpi) {
            autPostProcess();
        }
    };
    this.StopMonitoring = function () {
        stopTimer();
    };
}

function TestRunMessages(messageDiv, testRunMonitor, framework) {
    var TestStatusMessageCtxObj = function () {
        this.MessageType = {
            0: "MessageTypeInfo",
            1: "MessageTypeOutput",
            2: "MessageTypeError",
            3: "MessageTypeWarning",
            4: "MessageTypeCritical"
        };
        this.MessageSource = {
            0: "MessageSourceSetupScript",
            1: "MessageSourceCleanupScript",
            2: "MessageSourceValidation",
            3: "MessageSourceOther",
            4: "MessageSourceAutCounterCollection"
        };
        this.GridObject = null;
        this.AutoRefreshFunction = null;
    };
    var TestErrorsCtxObj = function () {
        this.GridObject = null;
        this.AutoRefreshFunction = null;
    };
    var testStatusMessageCtx;
    var testErrorsCtx;
    var selectedTreeView;
    var reopened = false;
    var stringsResource;
    var splitter;
    var TFSControls;
    var TFSGrids;
    var TFSCommonControls;
    var TFSCore;
    var globalRefreshIntervalId = null;
    var sampleRate = null;
    var instance = this;
    var treeView;
    var runStartedTime;
    this.globalGridCtx = null; /*USE THIS FOR GETTING CURRENTLY ACTIVE CTX*/

    this.Show = function () {
        if (!reopened) {
            if (!testRunMonitor.IsTestRunAvailable()) {
                window.setTimeout(instance.Show, 2000);
                return;
            }

            reopened = true;
            require(["Presentation/Scripts/TFS/TFS.UI.Controls", "Presentation/Scripts/TFS/TFS.UI.Controls.Common", "Presentation/Scripts/TFS/TFS.UI.Controls.Grids", "Presentation/Scripts/TFS/TFS.Core", "Presentation/Scripts/TFS/TFS.Core.Utils"], function (Controls, CommonControls, Grids, Core, Utils) {
                TFSControls = Controls;
                TFSGrids = Grids;
                TFSCommonControls = CommonControls;
                TFSCore = Core;
                TFSUtils = Utils;
                initializeView();
            });
        } else {
            //Refresh the current view on switch and set auto-refresh on.
            if (!framework.IsNullOrUndefined(instance.globalGridCtx)) {
                instance.globalGridCtx.AutoRefreshFunction();
            }
            refreshCurrentView();
        }
    };

    this.Hide = function () {
        if (globalRefreshIntervalId != null) {
            window.clearTimeout(globalRefreshIntervalId);
            globalRefreshIntervalId = null;
        }
    };

    var setSelectedItem = function (gridObj, selectedRowData) {
        var len = gridObj._dataSource.length;
        var rowLen = 4;
        for (var i = 0; i < len; i++) {
            var data = gridObj._dataSource[i];
            var flag = true;
            for (var j = 0; j < rowLen; j++) {
                if (selectedRowData[j] != data[j]) {
                    flag = false;
                    break;
                }
            }
            if (flag == true) {
                gridObj.setSelectedRowIndex(i);
                break;
            }
        }
    };

    var getSelectedGridRowData = function (gridObj) {
        var selectedRowIndex = gridObj.getSelectedRowIndex();
        var selectedRowData = null;
        if (selectedRowIndex >= 0)
            selectedRowData = gridObj._dataSource[selectedRowIndex];
        return selectedRowData;
    };

    var populateGrid = function (receivedData) {
        var selectedRowData = getSelectedGridRowData(instance.globalGridCtx.GridObject);
        instance.globalGridCtx.GridObject._options.source = receivedData;
        var sortOrder = instance.globalGridCtx.GridObject._options.sortOrder = instance.globalGridCtx.GridObject.getSortOrder();
        instance.globalGridCtx.GridObject._resetScroll = false;
        instance.globalGridCtx.GridObject.initializeDataSource();
        if (framework.IsNullOrUndefined(selectedRowData))
            selectedRowData = false;
        instance.globalGridCtx.GridObject.onSort(sortOrder, instance.globalGridCtx.GridObject._getSortColumns(sortOrder), selectedRowData);
        instance.globalGridCtx.GridObject.getSelectedRowIntoView();
    };

    var refreshCurrentView = function (messageCount) {
        refreshTreeView(messageCount);
        var testState = testRunMonitor.GetTestState();
        if (testState.IsFinished()) {
            stopAutoRefresh();
        } else {
            if (!framework.IsNullOrUndefined(instance.globalGridCtx)) {
                stopAutoRefresh();
                globalRefreshIntervalId = window.setTimeout(instance.globalGridCtx.AutoRefreshFunction, getRefreshFrequency());
            } else {
                createMessageGrid();
            }
        }
    };

    var getRefreshFrequency = function() {
        var defaultRefreshFrequency = 5000;
        
        if (!framework.IsNullOrUndefined(sampleRate)) {
            return (sampleRate * 1000);
        }
        
        var currentTest = testRunMonitor.GetTestRun();
        sampleRate = currentTest.RunSpecificDetails.SamplingInterval;

        if (!framework.IsNullOrUndefined(sampleRate)) {
            return (sampleRate * 1000);
        }

        return (defaultRefreshFrequency);
    };

    var initializeView = function () {
        stringsResource = framework.GetResource().GetString;
        buildPageHtml();
        createTreeView();
    };

    var overrideOnSort = function (gridObject) {
        gridObject.onSort = function (sortOrder, sortColumns, selectedRowData) {
            if (framework.IsNullOrUndefined(selectedRowData)) {
                selectedRowData = getSelectedGridRowData(gridObject);
            }
            gridObject._trySorting(sortOrder, sortColumns);
            gridObject._sortOrder = sortOrder;
            gridObject.layout();
            //In case we are loading for the first time
            if (selectedRowData == false)
                selectedRowData = getSelectedGridRowData(gridObject);
            setSelectedItem(gridObject, selectedRowData);
        };
    };

    var createMessageGrid = function () {
        //Create the appropriate message grid based on
        //which tree item is selected.
        stopAutoRefresh();
        if (!framework.IsNullOrUndefined(selectedTreeView)) {
            switch (selectedTreeView.key) {
                case stringsResource("StatusMessages"):
                    createStatusMessageGrid();
                    populateStatusMessageGrid();
                    break;
                default:
                    createTestErrorsGrid();
                    populateTestErrorsGrid();
                    break;
            }
        }
    };

    var getTestErrorGridData = function (receivedTestErrors) {
        var gridData = [];
        var typeIndex, subTypeIndex, detailsIndex;
        if (!framework.IsNullOrUndefined(receivedTestErrors.Types)) {
            for (typeIndex = 0; typeIndex < receivedTestErrors.Types.length; typeIndex++) {
                var errorType = receivedTestErrors.Types[typeIndex];
                if (!framework.IsNullOrUndefined(errorType.SubTypes)) {
                    for (subTypeIndex = 0; subTypeIndex < errorType.SubTypes.length; subTypeIndex++) {
                        var errorSubType = errorType.SubTypes[subTypeIndex];
                        if (!framework.IsNullOrUndefined(errorSubType.ErrorDetailList)) {
                            for (detailsIndex = 0; detailsIndex < errorSubType.ErrorDetailList.length; detailsIndex++) {
                                var error = errorSubType.ErrorDetailList[detailsIndex];
                                var gridDataItem = [];
                                //Error type
                                gridDataItem.push(returnHyphenOrValue(errorType.TypeName));
                                //Error subtype
                                gridDataItem.push(returnHyphenOrValue(errorSubType.SubTypeName));
                                //Error test
                                gridDataItem.push(returnHyphenOrValue(error.TestCaseName));
                                //Error request
                                gridDataItem.push(returnHyphenOrValue(error.Request));
                                //Error Scenario
                                gridDataItem.push(returnHyphenOrValue(error.ScenarioName));
                                //Error occurrences
                                gridDataItem.push(error.Occurrences);
                                //Error last time
                                var relativeTime = getRelativeTime(error.LastErrorDate);
                                gridDataItem.push(relativeTime);
                                //Error text
                                gridDataItem.push(error.MessageText);
                                //Error stack
                                gridDataItem.push(error.StackTrace);

                                gridData.push(gridDataItem);
                            }
                        }
                    }
                }
            }
        }
        return gridData;
    };

    var returnHyphenOrValue = function(item) {
        if (!framework.IsNullOrUndefined(item) && item.length > 0)
            return item;
        return "-";
    };

    var populateTestErrorsGrid = function () {
        var testRun = testRunMonitor.GetVSCompatibleTestRun();
        var treeNode = selectedTreeView;

        framework.GetTestServiceClient().GetTestErrors(testRun.Id, selectedTreeView.type, selectedTreeView.subtype, true,
            function (receivedTestErrors) {
                //This is async call, to prevent wrong grid population in case of quick switch between views 
                // we need to check treeNode is still selected.
                if (!framework.IsNullOrUndefined(receivedTestErrors) && treeNode.selected) {
                    var gridData = getTestErrorGridData(receivedTestErrors);
                    populateGrid(gridData);
                }
                refreshCurrentView();
            },
            function (error) {
                framework.LogError(error);
                refreshCurrentView();
            });
    };

    var createTestErrorsGrid = function () {
        $("#grid-host").html("");
        if (framework.IsNullOrUndefined(testErrorsCtx)) {
            //Initialize current test status messages context object
            testErrorsCtx = new TestErrorsCtxObj();
            var gridOptions = testErrorsGridOptions();
            var gridObject = TFSControls.BaseControl.createIn(TFSGrids.Grid, $("#grid-host"), gridOptions);

            gridObject._copySelectedItems = function () {
                gridObject._beginCopySelection(null, null);
            };
            gridObject._onContainerResize = function (e) {
                var columnWidth = gridColumnWidth();
                gridObject._options.columns[0].width = columnWidth;
                gridObject._options.columns[1].width = columnWidth * 2;
                gridObject._options.columns[2].width = columnWidth * 2;
                gridObject._options.columns[3].width = columnWidth * 4;
                gridObject._options.columns[4].width = columnWidth;
                gridObject._options.columns[5].width = columnWidth * .75;
                gridObject._options.columns[6].width = columnWidth * 2;
                gridObject._options.columns[7].width = columnWidth * 5;
                gridObject._options.columns[8].width = columnWidth * .75;
                this.layout();
            };
            overrideOnSort(gridObject);
            testErrorsCtx.GridObject = gridObject;
            testErrorsCtx.AutoRefreshFunction = populateTestErrorsGrid;
        } else {
            //Grid already created. Reuse existing grid.
            $("#grid-host").append($("<div>").attr("id", "new-grid"));
            testErrorsCtx.GridObject._enhance($("#new-grid"));
            testErrorsCtx.GridObject._options.sortOrder = testErrorsCtx.GridObject.getSortOrder();
            testErrorsCtx.GridObject.initialize();
        }
        //Clear and set global context for grid population.
        instance.globalGridCtx = null;
        instance.globalGridCtx = testErrorsCtx;
    };

    var gridColumnWidth = function () {
        var leftWidth = splitter.isExpanded() ? splitter._fixedSidePixels : 0;
        var columnWidth = (window.innerWidth - leftWidth) / 19;
        columnWidth = columnWidth < 50 ? 50 : columnWidth;
        return columnWidth;
    };

    var testErrorsGridOptions = function () {
        var options = {
            columns: testErrorColumns(),
            allowMultiSelect: false,
            keepSelection: true,
            gutter: {
                contextMenu: false
            },
            sortOrder: [
                {
                    index: 6,
                    order: "desc"
                }
            ]
        };
        return options;
    };

    var testErrorColumns = function () {
        var date = new Date();
        var columnWidth = gridColumnWidth();
        var timeMaxLength = date.toLocaleString().length;
        var columns = [
            { text: stringsResource("GridColumnType"), index: 0, width: columnWidth, canSortBy: false },
            { text: stringsResource("GridColumnSubtype"), index: 1, width: columnWidth * 2, canSortBy: false },
            { text: stringsResource("GridColumnTest"), index: 2, width: columnWidth * 2, canSortBy: false },
            { text: stringsResource("GridColumnRequest"), index: 3, width: columnWidth * 4, canSortBy: false },
            { text: stringsResource("GridColumnScenario"), index: 4, width: columnWidth, canSortBy: false },
            { text: stringsResource("GridColumnOccurrences"), index: 5, width: columnWidth * .75, canSortBy: false },
            { text: stringsResource("GridColumnLastTime"), index: 6, width: columnWidth * 2 },
            { text: stringsResource("GridColumnLastText"), index: 7, width: columnWidth * 5, canSortBy: false },
            {
                text: stringsResource("GridColumnLastStack"), index: 8, width: columnWidth * .75, canSortBy: false,
                getCellContents: function (rowInfo, dataIndex, expandedState, level, column) {
                    var stackTrace = this.getColumnValue(dataIndex, column.index);
                    if (framework.IsNullOrUndefined(stackTrace) || stackTrace.length == 0) {
                        stackTrace = stringsResource("NoStackTrace");
                    }
                    stackTrace = "<div UNSELECTABLE='off' style='word-wrap:break-word;'>" + stackTrace.replace("\n", "<br/>") + "</div>";
                    var stackLink = $("<a id='stackLink' href=#>" + stringsResource("StackLinkText") + "</a>");
                    stackLink.click(function (e) {
                        ShowBasicDialog($("#messageDetails"), stringsResource("LastStackDialogHeader"), [stackTrace]);
                    });
                    return $("<div class='grid-cell'/>").width(column.width || 100).append(stackLink);
                }
            }
        ];
        return columns;
    };

    var getRelativeTime = function (dateTimeString) {
        var timeSpan;
        var dateTime = new DateTime(dateTimeString, "ISO");
        var time = dateTime.GetJSDate().getTime();
        timeSpan = new TimeSpan(0);
        if (framework.IsNullOrUndefined(runStartedTime)) {
            var testRun = testRunMonitor.GetTestRun();
            if (!framework.IsNullOrUndefined(testRun)) {
                if (testRun.ExecutionStartedDate != null && testRun.ExecutionStartedDate !== testRun.CreatedDate) {
                    var executionStarted = new DateTime(testRun.ExecutionStartedDate, "ISO");
                    runStartedTime = executionStarted.GetJSDate().getTime();
                    //Refresh the current view on switch and set auto-refresh on.
                    if (!framework.IsNullOrUndefined(instance.globalGridCtx)) {
                        instance.globalGridCtx.AutoRefreshFunction();
                    }
                }
            }
        }
        if (!framework.IsNullOrUndefined(runStartedTime)) {
            var diff = time - runStartedTime;
            diff = diff < 0 ? 0 : diff;
            var seconds = Math.floor(diff / 1000);
            timeSpan = new TimeSpan(seconds);
        }
        return timeSpan.ToString();
    };

    var getStatusMessageGridData = function (receivedTestRunMessages) {
        var gridData = [];
        $.each(receivedTestRunMessages, function (index, message) {
            var gridDataItem = [];
            //Message level
            var messageType = message.MessageType.toString();

            gridDataItem.push(messageType);
            //Message date and time difference from run start time
            var relativeTime = getRelativeTime(message.LoggedDate);
            gridDataItem.push(relativeTime);
            //Message type
            var messageSource = message.MessageSource.toString();
            gridDataItem.push(stringsResource(instance.globalGridCtx.MessageSource[messageSource]));
            //Message source
            if (framework.IsNullOrUndefined(message.AgentId) || message.AgentId.length == 0) {
                gridDataItem.push("-");
            } else {
                gridDataItem.push(message.AgentId);
            }
            //Message body
            gridDataItem.push({
                ErrorCode: message.ErrorCode,
                Message: message.Message,
                Url: message.Url
            });

            //Blank for view details link
            gridDataItem.push("");

            gridData.push(gridDataItem);
        });
        return gridData;
    };

    var populateStatusMessageGrid = function () {
        var testRun = testRunMonitor.GetVSCompatibleTestRun();
        var treeNode = selectedTreeView;
        var messages = testRunMonitor.GetTestMessages();
        if (!framework.IsNullOrUndefined(messages)) {
            var gridData = getStatusMessageGridData(messages);
            populateGrid(gridData);
            refreshCurrentView(messages.length);
        }
    };

    var createStatusMessageGrid = function () {
        $("#grid-host").html("");
        if (framework.IsNullOrUndefined(testStatusMessageCtx)) {
            //Initialize current test status messages context object
            testStatusMessageCtx = new TestStatusMessageCtxObj();
            var gridOptions = statusMessageGridOptions(testStatusMessageCtx);
            var gridObject = TFSControls.BaseControl.createIn(TFSGrids.Grid, $("#grid-host"), gridOptions);
            gridObject._onContainerResize = function (e) {
                var columnWidth = gridColumnWidth();
                gridObject._options.columns[0].width = columnWidth * 2;
                gridObject._options.columns[1].width = columnWidth * 2;
                gridObject._options.columns[2].width = columnWidth * 2;
                gridObject._options.columns[3].width = columnWidth;
                gridObject._options.columns[4].width = columnWidth * 10;
                gridObject._options.columns[5].width = columnWidth * 2;
                this.layout();
            };
            gridObject._copySelectedItems = function () {
                gridObject._beginCopySelection(null, null);
            };
            overrideOnSort(gridObject);
            testStatusMessageCtx.GridObject = gridObject;
            testStatusMessageCtx.AutoRefreshFunction = populateStatusMessageGrid;
        } else {
            //Grid already created. Reuse existing grid.
            $("#grid-host").append($("<div>").attr("id", "new-grid"));
            testStatusMessageCtx.GridObject._enhance($("#new-grid"));
            testStatusMessageCtx.GridObject._options.sortOrder = testStatusMessageCtx.GridObject.getSortOrder();
            testStatusMessageCtx.GridObject.initialize();
        }
        //Clear and set global context for grid population.
        instance.globalGridCtx = null;
        instance.globalGridCtx = testStatusMessageCtx;
    };

    var statusMessageGridOptions = function (testStatusMsgCtx) {
        var options = {
            columns: statusMessageColumns(testStatusMsgCtx),
            allowMultiSelect: false,
            keepSelection: true,
            gutter: {
                contextMenu: false
            },
            sortOrder: [
                {
                    index: 1,
                    order: "desc"
                }
            ]
        };
        return options;
    };

    var statusMessageColumns = function (testStatusMsgCtx) {
        var date = new Date();
        var timeMaxLength = date.toLocaleString().length;
        var columnWidth = gridColumnWidth();
        var columns = [
            {
                text: stringsResource("GridColumnLevel"), index: 0, width: columnWidth * 2,
                getCellContents: function (rowInfo, dataIndex, expandedState, level, column) {
                    var messageType = this.getColumnValue(dataIndex, column.index);
                    var messageImageStyle = {
                        0: "iconInfoGrid",
                        1: "iconInfoGrid",
                        2: "iconErrorGrid",
                        3: "iconWarningGrid",
                        4: "iconCriticalGrid"
                    };
                    return $("<div class='grid-cell'/>").width(column.width).html("<img class='" + messageImageStyle[messageType] + "' data-plugin-theme-src='" +
                        messageImageStyle[messageType] + "' src='" + Plugin.Theme.getValue(messageImageStyle[messageType]) + "'/>  " +
                        "<span>" + stringsResource(testStatusMsgCtx.MessageType[messageType]) + "</span>");
                }
            },
            { text: stringsResource("GridColumnTime"), index: 1, width: columnWidth * 2 },
            { text: stringsResource("GridColumnType"), index: 2, width: columnWidth * 2 },
            { text: stringsResource("GridColumnSource"), index: 3, width: columnWidth },
            {
                text: stringsResource("GridColumnMessage"), index: 4, width: columnWidth * 10, canSortBy: false, fixed: true,
                getCellContents: function (rowInfo, dataIndex, expandedState, level, column) {
                    var messageData = this.getColumnValue(dataIndex, column.index);
                    if (!framework.IsNullOrUndefined(messageData.ErrorCode) && !framework.IsNullOrUndefined(messageData.Url)) {
                        var errorCodeLink = $("<a class='messageErrorCode' href=" + messageData.Url + ">" + messageData.ErrorCode + ": " + "</a>");
                        errorCodeLink.click(function (e) {
                            framework.OpenAutHelpLink($(this).attr('href'));
                            e.preventDefault();
                        });
                        return $("<div class='grid-cell'/>").width(column.width).append(errorCodeLink).append("<span>" + messageData.Message + "</span>");
                    } else {
                        return $("<div class='grid-cell'/>").width(column.width).html("<span>" + messageData.Message + "</span>");
                    }
                }
            },
            {
                text: "", index: 5, width: columnWidth * 2, canSortBy: false, fixed: true, maxLength: stringsResource("ViewDetails").length,
                getCellContents: function (rowInfo, dataIndex, expandedState, level, column) {
                    var messageData = this.getColumnValue(dataIndex, column.index - 1);
                    var messageDiv;
                    var messageWithBreaks = messageData.Message.replace("\n", "<br/>");
                    if (!framework.IsNullOrUndefined(messageData.ErrorCode) && !framework.IsNullOrUndefined(messageData.Url)) {
                        var errorCodeLink = $("<a class='messageDialogErrorCode' href=" + messageData.Url + ">" + messageData.ErrorCode + ": " + "</a>");
                        messageDiv = $("<span UNSELECTABLE='off'/>").width(column.width).append(errorCodeLink).append("<span UNSELECTABLE='off'>" + messageWithBreaks + "</span>");
                    } else {
                        messageDiv = $("<span UNSELECTABLE='off'/>").width(column.width).html("<span UNSELECTABLE='off'>" + messageWithBreaks + "</span>");
                    }

                    var viewDetailsLink = $("<a id='messageLink' href=#>" + stringsResource("ViewDetails") + "</a>");
                    viewDetailsLink.click(function (e) {
                        ShowElementsDialog($("#messageDetails"), stringsResource("GridColumnMessage"), messageDiv);
                    });
                    return $("<div class='grid-cell'/>").width(column.width).append(viewDetailsLink);
                }
            }
        ];
        return columns;
    };

    $("#messageDetails").delegate(".messageDialogErrorCode", "click",
            function (event) {
                framework.OpenAutHelpLink($(this).attr('href'));
                event.preventDefault();
            });

    var createTreeView = function () {

        TFSControls.TreeNode.prototype.queryStringParams = "";
        TFSControls.TreeNode.prototype.key = "";
        TFSControls.TreeNode.prototype.count = 0;

        var $treeElem = $("#MessagesTreeView");
        var messagesNode;
        var testRun = testRunMonitor.GetVSCompatibleTestRun();
        framework.GetTestServiceClient().GetTestRunMessages(testRun.Id, function (messages) {
            messagesNode = new TFSControls.TreeNode(stringsResource("StatusMessages"));
            messagesNode.key = stringsResource("StatusMessages");
            if (!framework.IsNullOrUndefined(messages.length))
                messagesNode.count = messages.length;
            else {
                messagesNode.count = 0;
            }
        },
        function (error) {
            framework.LogError(error);
        }
        );

        framework.GetTestServiceClient().GetTestErrors(testRun.Id, null, null, false,
            function (receivedTestErrors) {
                var errorNode = convertToTreeNodes(receivedTestErrors);
                var messageTreeOptions = {
                    width: "100%",
                    height: "100%",
                    nodes: [messagesNode, errorNode],
                };
                $treeElem.html("");
                treeView = TFSControls.BaseControl.createIn(TFSCommonControls.TreeView, $treeElem, messageTreeOptions);
                treeView._onToggle = function (e) {
                    var li = $(e.target).parents("li.node:first");
                    treeView._toggle(this._getNode(li), li);
                    treeView._draw();
                    return false;
                };
                treeView.onItemClick = function (selectedTreeItem) {
                    treeView.setSelectedNode(selectedTreeItem);
                    if (selectedTreeView != treeView.getSelectedNode()) {
                        selectedTreeView = treeView.getSelectedNode();
                        splitter.setCollapsedLabel(selectedTreeView.key);
                        createMessageGrid();
                    }
                };
                if (framework.IsNullOrUndefined(selectedTreeView) && (!framework.IsNullOrUndefined(messageTreeOptions.nodes))) {
                    treeView.setSelectedNode(messageTreeOptions.nodes[0]);
                    selectedTreeView = treeView.getSelectedNode();
                    splitter.setCollapsedLabel(selectedTreeView.key);
                    createMessageGrid();
                }
                hideLoading();
            },
            function (error) {
                framework.LogError(error);
                hideLoading();
            });
    };

    var refreshTreeView = function (messageCount) {
        var testRun = testRunMonitor.GetVSCompatibleTestRun();
        var messagesNode = new TFSControls.TreeNode(stringsResource("StatusMessages"));
        messagesNode.key = stringsResource("StatusMessages");
        if (!framework.IsNullOrUndefined(messageCount)) {
            messagesNode.count = messageCount;
        }
        else {
            var messages = testRunMonitor.GetTestMessages();
            if (!framework.IsNullOrUndefined(messages)) {
                messagesNode.count = messages.length;
            }
            else {
                messagesNode.count = 0;
            }
        }

        framework.GetTestServiceClient().GetTestErrors(testRun.Id, null, null, false,
            function (receivedTestErrors) {
                var errorNode = convertToTreeNodes(receivedTestErrors);
                var nodes = [messagesNode, errorNode];
                if (!framework.IsNullOrUndefined(nodes)) {
                    if (framework.IsNullOrUndefined(treeView)) {
                        framework.LogWarning("treeView is not created yet, waiting for it");
                    } else {
                        $.map(nodes, function (node) {
                            updateTreeView(treeView.rootNode, node);
                        });

                        treeView._draw();
                    }
                }
            });
    };

    var updateTreeView = function (root, newNode) {
        if (framework.IsNullOrUndefined(root) || framework.IsNullOrUndefined(newNode)) return;
        var foundNode = TFSUtils.findTreeNode.call(root, newNode.text, "/", TFSCore.StringUtils.localeIgnoreCaseComparer, "key");
        if (framework.IsNullOrUndefined(foundNode)) {
            root.add(newNode);
            foundNode = newNode;
        }
        foundNode.text = newNode.text + " (" + newNode.count + ")";
        $.map(newNode.children, function (child) {
            updateTreeView(foundNode, child);
        });
    };

    var convertToTreeNodes = function (errorDetail) {
        var node = new TFSControls.TreeNode(stringsResource("ErrorsExceptions"));
        node.key = stringsResource("ErrorsExceptions");
        node.count = 0;
        if (!framework.IsNullOrUndefined(errorDetail)) {
            node.count = errorDetail.Occurrences;
            node.type = null;
            node.subtype = null;
            if (!framework.IsNullOrUndefined(errorDetail.Types)) {
                errorDetail.Types.forEach(function (item) {
                    if (!framework.IsNullOrUndefined(item.TypeName) && item.TypeName.length > 0) {
                        var childNode = convertToErrorTypeNodes(item);
                        node.add(childNode);
                    }
                });
            }

        }
        return node;
    };

    var convertToErrorTypeNodes = function(errorTypeDetail) {
        var node = new TFSControls.TreeNode(errorTypeDetail.TypeName);
        node.key = errorTypeDetail.TypeName;
        node.count = errorTypeDetail.Occurrences;
        node.type = errorTypeDetail.TypeName;
        node.subtype = null;
        if (!framework.IsNullOrUndefined(errorTypeDetail.SubTypes)) {
            errorTypeDetail.SubTypes.forEach(function(item) {
                if (!framework.IsNullOrUndefined(item.SubTypeName) && item.SubTypeName.length > 0) {
                    var childNode = convertToErrorSubTypeNodes(item, errorTypeDetail.TypeName);
                    node.add(childNode);
                }
            });
        }
        return node;
    };

    var convertToErrorSubTypeNodes = function(errorSubTypeDetail, typeName) {
        var node = new TFSControls.TreeNode(errorSubTypeDetail.SubTypeName);
        node.key = errorSubTypeDetail.SubTypeName;
        node.count = errorSubTypeDetail.Occurrences;
        node.type = typeName;
        node.subtype = errorSubTypeDetail.SubTypeName;
        return node;
    };


    var stopAutoRefresh = function () {
        instance.Hide();
    };

    var buildPageHtml = function () {
        splitter = TFSControls.BaseControl.createIn(TFSCommonControls.Splitter, messageDiv,
                   { enableToggleButton: true, vertical: false, fixedSide: "left", cssClass: "splitter", collapsedLabel: stringsResource("MessageTabLabel") });
        var $leftPanelHtml = $("<div>").addClass("left-hub-content").append($("<div>").attr("id", "MessagesTreeView"));
        splitter.leftPane.append($leftPanelHtml);
        var $rightPanelHtml = "<div class='details-progress hub-progress pageProgressIndicator' style='display: none;'></div>\
                               <div class='right-hub-content'><div class='hub-pivot-content'>\
                               </div><div id='grid-host'></div></div>";
        splitter.rightPane.html($rightPanelHtml);
        splitter._element.addClass("hub-splitter divided-section toggle-button-enabled toggle-button-hotkey-enabled");
        splitter.resize("250px", true);
        showLoading();
    };

    var showLoading = function () {
        $(".pageProgressIndicator").show();
    };

    var hideLoading = function () {
        $(".pageProgressIndicator").hide();
    };
}
// SIG // Begin signature block
// SIG // MIIarwYJKoZIhvcNAQcCoIIaoDCCGpwCAQExCzAJBgUr
// SIG // DgMCGgUAMGcGCisGAQQBgjcCAQSgWTBXMDIGCisGAQQB
// SIG // gjcCAR4wJAIBAQQQEODJBs441BGiowAQS9NQkAIBAAIB
// SIG // AAIBAAIBAAIBADAhMAkGBSsOAwIaBQAEFEUD5Xagy2Sm
// SIG // /eGvGL6f584IniMyoIIVgjCCBMMwggOroAMCAQICEzMA
// SIG // AABxsy6Ka4KqH04AAAAAAHEwDQYJKoZIhvcNAQEFBQAw
// SIG // dzELMAkGA1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0
// SIG // b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1p
// SIG // Y3Jvc29mdCBDb3Jwb3JhdGlvbjEhMB8GA1UEAxMYTWlj
// SIG // cm9zb2Z0IFRpbWUtU3RhbXAgUENBMB4XDTE1MDMyMDE3
// SIG // MzIwM1oXDTE2MDYyMDE3MzIwM1owgbMxCzAJBgNVBAYT
// SIG // AlVTMRMwEQYDVQQIEwpXYXNoaW5ndG9uMRAwDgYDVQQH
// SIG // EwdSZWRtb25kMR4wHAYDVQQKExVNaWNyb3NvZnQgQ29y
// SIG // cG9yYXRpb24xDTALBgNVBAsTBE1PUFIxJzAlBgNVBAsT
// SIG // Hm5DaXBoZXIgRFNFIEVTTjpCOEVDLTMwQTQtNzE0NDEl
// SIG // MCMGA1UEAxMcTWljcm9zb2Z0IFRpbWUtU3RhbXAgU2Vy
// SIG // dmljZTCCASIwDQYJKoZIhvcNAQEBBQADggEPADCCAQoC
// SIG // ggEBAOqRvbKI/RRvITYoA2YzOmYI+1tLpKugKDRKQzII
// SIG // wIblyT3VJbx7PmKH1n3vD3RTo/GRY4h0f+gkzQNQxfHK
// SIG // ABZ7pTmwBhw8RH7568SygbwXI7r9ZTgZhX/KoCn99jrA
// SIG // Cy9o9OA0Tn1vF8Bumar6f2El0SZw0nR932FzXM5UKjlR
// SIG // AzMJ+FCteMeJCLbUhSo/19gfUerv/GhetcHnB2gyjS9y
// SIG // Uf4DMUdRxdLrcgevIJX42mr4d2fkYJpwTKtFy34Ir+WB
// SIG // 1FfPOswTdZ0mzaCiaVC8OoiU37BUON6JOc2GMqWQD36/
// SIG // 7cyUJaZBhmEmx903flwN6BfKN3/oJLZOtPgbI+sCAwEA
// SIG // AaOCAQkwggEFMB0GA1UdDgQWBBT4/SOHBZSAVs0zpUHC
// SIG // bMwINsiyojAfBgNVHSMEGDAWgBQjNPjZUkZwCu1A+3b7
// SIG // syuwwzWzDzBUBgNVHR8ETTBLMEmgR6BFhkNodHRwOi8v
// SIG // Y3JsLm1pY3Jvc29mdC5jb20vcGtpL2NybC9wcm9kdWN0
// SIG // cy9NaWNyb3NvZnRUaW1lU3RhbXBQQ0EuY3JsMFgGCCsG
// SIG // AQUFBwEBBEwwSjBIBggrBgEFBQcwAoY8aHR0cDovL3d3
// SIG // dy5taWNyb3NvZnQuY29tL3BraS9jZXJ0cy9NaWNyb3Nv
// SIG // ZnRUaW1lU3RhbXBQQ0EuY3J0MBMGA1UdJQQMMAoGCCsG
// SIG // AQUFBwMIMA0GCSqGSIb3DQEBBQUAA4IBAQAtBLTKKQtZ
// SIG // /C7qoK9MTmgE+JLtKcJmzGtwyYfovof8XfTdT6Uab3iX
// SIG // rWsFOFFBcp055Bobw21x/HC208y2kFgEKD/WHu+DsxQY
// SIG // DJUL96URE5jGhVZe7jO0DDe1gOr1EmjZLnuGCHI7FHvU
// SIG // 2dAWT8AvCx8tyuUb0K7phLCPC11zuBaBQCNYLOphqv69
// SIG // f9ONWnD8ec1mlmVjtQUSduIqOyvtgqya7CdBp5cOIxaf
// SIG // QchObVMRQATMYJnamOwrrpf74H31uosA9CUXf2J6u1FX
// SIG // wfDwzZwbYXOtlYwrdiKoq3A4tAEofWZCU96f9Ad8WjAO
// SIG // ggNZ9oSGuRUlYrAL0s/x25ZFMIIE7DCCA9SgAwIBAgIT
// SIG // MwAAAQosea7XeXumrAABAAABCjANBgkqhkiG9w0BAQUF
// SIG // ADB5MQswCQYDVQQGEwJVUzETMBEGA1UECBMKV2FzaGlu
// SIG // Z3RvbjEQMA4GA1UEBxMHUmVkbW9uZDEeMBwGA1UEChMV
// SIG // TWljcm9zb2Z0IENvcnBvcmF0aW9uMSMwIQYDVQQDExpN
// SIG // aWNyb3NvZnQgQ29kZSBTaWduaW5nIFBDQTAeFw0xNTA2
// SIG // MDQxNzQyNDVaFw0xNjA5MDQxNzQyNDVaMIGDMQswCQYD
// SIG // VQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4G
// SIG // A1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0
// SIG // IENvcnBvcmF0aW9uMQ0wCwYDVQQLEwRNT1BSMR4wHAYD
// SIG // VQQDExVNaWNyb3NvZnQgQ29ycG9yYXRpb24wggEiMA0G
// SIG // CSqGSIb3DQEBAQUAA4IBDwAwggEKAoIBAQCS/G82u+ED
// SIG // uSjWRtGiYbqlRvtjFj4u+UfSx+ztx5mxJlF1vdrMDwYU
// SIG // EaRsGZ7AX01UieRNUNiNzaFhpXcTmhyn7Q1096dWeego
// SIG // 91PSsXpj4PWUl7fs2Uf4bD3zJYizvArFBKeOfIVIdhxh
// SIG // RqoZxHpii8HCNar7WG/FYwuTSTCBG3vff3xPtEdtX3gc
// SIG // r7b3lhNS77nRTTnlc95ITjwUqpcNOcyLUeFc0Tvwjmfq
// SIG // MGCpTVqdQ73bI7rAD9dLEJ2cTfBRooSq5JynPdaj7woY
// SIG // SKj6sU6lmA5Lv/AU8wDIsEjWW/4414kRLQW6QwJPIgCW
// SIG // Ja19NW6EaKsgGDgo/hyiELGlAgMBAAGjggFgMIIBXDAT
// SIG // BgNVHSUEDDAKBggrBgEFBQcDAzAdBgNVHQ4EFgQUif4K
// SIG // MeomzeZtx5GRuZSMohhhNzQwUQYDVR0RBEowSKRGMEQx
// SIG // DTALBgNVBAsTBE1PUFIxMzAxBgNVBAUTKjMxNTk1KzA0
// SIG // MDc5MzUwLTE2ZmEtNGM2MC1iNmJmLTlkMmIxY2QwNTk4
// SIG // NDAfBgNVHSMEGDAWgBTLEejK0rQWWAHJNy4zFha5TJoK
// SIG // HzBWBgNVHR8ETzBNMEugSaBHhkVodHRwOi8vY3JsLm1p
// SIG // Y3Jvc29mdC5jb20vcGtpL2NybC9wcm9kdWN0cy9NaWND
// SIG // b2RTaWdQQ0FfMDgtMzEtMjAxMC5jcmwwWgYIKwYBBQUH
// SIG // AQEETjBMMEoGCCsGAQUFBzAChj5odHRwOi8vd3d3Lm1p
// SIG // Y3Jvc29mdC5jb20vcGtpL2NlcnRzL01pY0NvZFNpZ1BD
// SIG // QV8wOC0zMS0yMDEwLmNydDANBgkqhkiG9w0BAQUFAAOC
// SIG // AQEApqhTkd87Af5hXQZa62bwDNj32YTTAFEOENGk0Rco
// SIG // 54wzOCvYQ8YDi3XrM5L0qeJn/QLbpR1OQ0VdG0nj4E8W
// SIG // 8H6P8IgRyoKtpPumqV/1l2DIe8S/fJtp7R+CwfHNjnhL
// SIG // YvXXDRzXUxLWllLvNb0ZjqBAk6EKpS0WnMJGdAjr2/TY
// SIG // pUk2VBIRVQOzexb7R/77aPzARVziPxJ5M6LvgsXeQBkH
// SIG // 7hXFCptZBUGp0JeegZ4DW/xK4xouBaxQRy+M+nnYHiD4
// SIG // BfspaxgU+nIEtwunmmTsEV1PRUmNKRot+9C2CVNfNJTg
// SIG // FsS56nM16Ffv4esWwxjHBrM7z2GE4rZEiZSjhjCCBbww
// SIG // ggOkoAMCAQICCmEzJhoAAAAAADEwDQYJKoZIhvcNAQEF
// SIG // BQAwXzETMBEGCgmSJomT8ixkARkWA2NvbTEZMBcGCgmS
// SIG // JomT8ixkARkWCW1pY3Jvc29mdDEtMCsGA1UEAxMkTWlj
// SIG // cm9zb2Z0IFJvb3QgQ2VydGlmaWNhdGUgQXV0aG9yaXR5
// SIG // MB4XDTEwMDgzMTIyMTkzMloXDTIwMDgzMTIyMjkzMlow
// SIG // eTELMAkGA1UEBhMCVVMxEzARBgNVBAgTCldhc2hpbmd0
// SIG // b24xEDAOBgNVBAcTB1JlZG1vbmQxHjAcBgNVBAoTFU1p
// SIG // Y3Jvc29mdCBDb3Jwb3JhdGlvbjEjMCEGA1UEAxMaTWlj
// SIG // cm9zb2Z0IENvZGUgU2lnbmluZyBQQ0EwggEiMA0GCSqG
// SIG // SIb3DQEBAQUAA4IBDwAwggEKAoIBAQCycllcGTBkvx2a
// SIG // YCAgQpl2U2w+G9ZvzMvx6mv+lxYQ4N86dIMaty+gMuz/
// SIG // 3sJCTiPVcgDbNVcKicquIEn08GisTUuNpb15S3GbRwfa
// SIG // /SXfnXWIz6pzRH/XgdvzvfI2pMlcRdyvrT3gKGiXGqel
// SIG // cnNW8ReU5P01lHKg1nZfHndFg4U4FtBzWwW6Z1KNpbJp
// SIG // L9oZC/6SdCnidi9U3RQwWfjSjWL9y8lfRjFQuScT5EAw
// SIG // z3IpECgixzdOPaAyPZDNoTgGhVxOVoIoKgUyt0vXT2Pn
// SIG // 0i1i8UU956wIAPZGoZ7RW4wmU+h6qkryRs83PDietHdc
// SIG // pReejcsRj1Y8wawJXwPTAgMBAAGjggFeMIIBWjAPBgNV
// SIG // HRMBAf8EBTADAQH/MB0GA1UdDgQWBBTLEejK0rQWWAHJ
// SIG // Ny4zFha5TJoKHzALBgNVHQ8EBAMCAYYwEgYJKwYBBAGC
// SIG // NxUBBAUCAwEAATAjBgkrBgEEAYI3FQIEFgQU/dExTtMm
// SIG // ipXhmGA7qDFvpjy82C0wGQYJKwYBBAGCNxQCBAweCgBT
// SIG // AHUAYgBDAEEwHwYDVR0jBBgwFoAUDqyCYEBWJ5flJRP8
// SIG // KuEKU5VZ5KQwUAYDVR0fBEkwRzBFoEOgQYY/aHR0cDov
// SIG // L2NybC5taWNyb3NvZnQuY29tL3BraS9jcmwvcHJvZHVj
// SIG // dHMvbWljcm9zb2Z0cm9vdGNlcnQuY3JsMFQGCCsGAQUF
// SIG // BwEBBEgwRjBEBggrBgEFBQcwAoY4aHR0cDovL3d3dy5t
// SIG // aWNyb3NvZnQuY29tL3BraS9jZXJ0cy9NaWNyb3NvZnRS
// SIG // b290Q2VydC5jcnQwDQYJKoZIhvcNAQEFBQADggIBAFk5
// SIG // Pn8mRq/rb0CxMrVq6w4vbqhJ9+tfde1MOy3XQ60L/svp
// SIG // LTGjI8x8UJiAIV2sPS9MuqKoVpzjcLu4tPh5tUly9z7q
// SIG // QX/K4QwXaculnCAt+gtQxFbNLeNK0rxw56gNogOlVuC4
// SIG // iktX8pVCnPHz7+7jhh80PLhWmvBTI4UqpIIck+KUBx3y
// SIG // 4k74jKHK6BOlkU7IG9KPcpUqcW2bGvgc8FPWZ8wi/1wd
// SIG // zaKMvSeyeWNWRKJRzfnpo1hW3ZsCRUQvX/TartSCMm78
// SIG // pJUT5Otp56miLL7IKxAOZY6Z2/Wi+hImCWU4lPF6H0q7
// SIG // 0eFW6NB4lhhcyTUWX92THUmOLb6tNEQc7hAVGgBd3TVb
// SIG // Ic6YxwnuhQ6MT20OE049fClInHLR82zKwexwo1eSV32U
// SIG // jaAbSANa98+jZwp0pTbtLS8XyOZyNxL0b7E8Z4L5UrKN
// SIG // MxZlHg6K3RDeZPRvzkbU0xfpecQEtNP7LN8fip6sCvsT
// SIG // J0Ct5PnhqX9GuwdgR2VgQE6wQuxO7bN2edgKNAltHIAx
// SIG // H+IOVN3lofvlRxCtZJj/UBYufL8FIXrilUEnacOTj5XJ
// SIG // jdibIa4NXJzwoq6GaIMMai27dmsAHZat8hZ79haDJLmI
// SIG // z2qoRzEvmtzjcT3XAH5iR9HOiMm4GPoOco3Boz2vAkBq
// SIG // /2mbluIQqBC0N1AI1sM9MIIGBzCCA++gAwIBAgIKYRZo
// SIG // NAAAAAAAHDANBgkqhkiG9w0BAQUFADBfMRMwEQYKCZIm
// SIG // iZPyLGQBGRYDY29tMRkwFwYKCZImiZPyLGQBGRYJbWlj
// SIG // cm9zb2Z0MS0wKwYDVQQDEyRNaWNyb3NvZnQgUm9vdCBD
// SIG // ZXJ0aWZpY2F0ZSBBdXRob3JpdHkwHhcNMDcwNDAzMTI1
// SIG // MzA5WhcNMjEwNDAzMTMwMzA5WjB3MQswCQYDVQQGEwJV
// SIG // UzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4GA1UEBxMH
// SIG // UmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0IENvcnBv
// SIG // cmF0aW9uMSEwHwYDVQQDExhNaWNyb3NvZnQgVGltZS1T
// SIG // dGFtcCBQQ0EwggEiMA0GCSqGSIb3DQEBAQUAA4IBDwAw
// SIG // ggEKAoIBAQCfoWyx39tIkip8ay4Z4b3i48WZUSNQrc7d
// SIG // GE4kD+7Rp9FMrXQwIBHrB9VUlRVJlBtCkq6YXDAm2gBr
// SIG // 6Hu97IkHD/cOBJjwicwfyzMkh53y9GccLPx754gd6udO
// SIG // o6HBI1PKjfpFzwnQXq/QsEIEovmmbJNn1yjcRlOwhtDl
// SIG // KEYuJ6yGT1VSDOQDLPtqkJAwbofzWTCd+n7Wl7PoIZd+
// SIG // +NIT8wi3U21StEWQn0gASkdmEScpZqiX5NMGgUqi+YSn
// SIG // EUcUCYKfhO1VeP4Bmh1QCIUAEDBG7bfeI0a7xC1Un68e
// SIG // eEExd8yb3zuDk6FhArUdDbH895uyAc4iS1T/+QXDwiAL
// SIG // AgMBAAGjggGrMIIBpzAPBgNVHRMBAf8EBTADAQH/MB0G
// SIG // A1UdDgQWBBQjNPjZUkZwCu1A+3b7syuwwzWzDzALBgNV
// SIG // HQ8EBAMCAYYwEAYJKwYBBAGCNxUBBAMCAQAwgZgGA1Ud
// SIG // IwSBkDCBjYAUDqyCYEBWJ5flJRP8KuEKU5VZ5KShY6Rh
// SIG // MF8xEzARBgoJkiaJk/IsZAEZFgNjb20xGTAXBgoJkiaJ
// SIG // k/IsZAEZFgltaWNyb3NvZnQxLTArBgNVBAMTJE1pY3Jv
// SIG // c29mdCBSb290IENlcnRpZmljYXRlIEF1dGhvcml0eYIQ
// SIG // ea0WoUqgpa1Mc1j0BxMuZTBQBgNVHR8ESTBHMEWgQ6BB
// SIG // hj9odHRwOi8vY3JsLm1pY3Jvc29mdC5jb20vcGtpL2Ny
// SIG // bC9wcm9kdWN0cy9taWNyb3NvZnRyb290Y2VydC5jcmww
// SIG // VAYIKwYBBQUHAQEESDBGMEQGCCsGAQUFBzAChjhodHRw
// SIG // Oi8vd3d3Lm1pY3Jvc29mdC5jb20vcGtpL2NlcnRzL01p
// SIG // Y3Jvc29mdFJvb3RDZXJ0LmNydDATBgNVHSUEDDAKBggr
// SIG // BgEFBQcDCDANBgkqhkiG9w0BAQUFAAOCAgEAEJeKw1wD
// SIG // RDbd6bStd9vOeVFNAbEudHFbbQwTq86+e4+4LtQSooxt
// SIG // YrhXAstOIBNQmd16QOJXu69YmhzhHQGGrLt48ovQ7DsB
// SIG // 7uK+jwoFyI1I4vBTFd1Pq5Lk541q1YDB5pTyBi+FA+mR
// SIG // KiQicPv2/OR4mS4N9wficLwYTp2OawpylbihOZxnLcVR
// SIG // DupiXD8WmIsgP+IHGjL5zDFKdjE9K3ILyOpwPf+FChPf
// SIG // wgphjvDXuBfrTot/xTUrXqO/67x9C0J71FNyIe4wyrt4
// SIG // ZVxbARcKFA7S2hSY9Ty5ZlizLS/n+YWGzFFW6J1wlGys
// SIG // OUzU9nm/qhh6YinvopspNAZ3GmLJPR5tH4LwC8csu89D
// SIG // s+X57H2146SodDW4TsVxIxImdgs8UoxxWkZDFLyzs7BN
// SIG // Z8ifQv+AeSGAnhUwZuhCEl4ayJ4iIdBD6Svpu/RIzCzU
// SIG // 2DKATCYqSCRfWupW76bemZ3KOm+9gSd0BhHudiG/m4LB
// SIG // J1S2sWo9iaF2YbRuoROmv6pH8BJv/YoybLL+31HIjCPJ
// SIG // Zr2dHYcSZAI9La9Zj7jkIeW1sMpjtHhUBdRBLlCslLCl
// SIG // eKuzoJZ1GtmShxN1Ii8yqAhuoFuMJb+g74TKIdbrHk/J
// SIG // mu5J4PcBZW+JC33Iacjmbuqnl84xKf8OxVtc2E0bodj6
// SIG // L54/LlUWa8kTo/0xggSZMIIElQIBATCBkDB5MQswCQYD
// SIG // VQQGEwJVUzETMBEGA1UECBMKV2FzaGluZ3RvbjEQMA4G
// SIG // A1UEBxMHUmVkbW9uZDEeMBwGA1UEChMVTWljcm9zb2Z0
// SIG // IENvcnBvcmF0aW9uMSMwIQYDVQQDExpNaWNyb3NvZnQg
// SIG // Q29kZSBTaWduaW5nIFBDQQITMwAAAQosea7XeXumrAAB
// SIG // AAABCjAJBgUrDgMCGgUAoIGyMBkGCSqGSIb3DQEJAzEM
// SIG // BgorBgEEAYI3AgEEMBwGCisGAQQBgjcCAQsxDjAMBgor
// SIG // BgEEAYI3AgEVMCMGCSqGSIb3DQEJBDEWBBTVv7OWv3F2
// SIG // eqxJ8lt/cJljRrD8NDBSBgorBgEEAYI3AgEMMUQwQqAo
// SIG // gCYARQBMAFMALgBUAGUAcwB0AFIAdQBuAEcAcgBhAHAA
// SIG // aAAuAGoAc6EWgBRodHRwOi8vbWljcm9zb2Z0LmNvbTAN
// SIG // BgkqhkiG9w0BAQEFAASCAQARBsp6IhoyM0uBBXimHF+7
// SIG // vpR5v9Epbdo+Y1MoRs33zvef/mFy1Ja8RMfnqaMenwzp
// SIG // B0oVvDsYX/LfASQgdsk82A9oFixedEWjSxgfRGKWFppc
// SIG // npt0UywtR4mbrwkyOIpfjZZrd6YOGRBnNmJiqEn2YpEc
// SIG // ZJF0flg5kXZecYdnM9PERY0NTy1va61GXxOWZGmicA2P
// SIG // 8aeb5QgMohr/WJYSdgVLfTPezHus4H8vf/uvyQlOs4ir
// SIG // tBcTPamwqShSUMkT2vQ3yT5y02QiBd4WJHKaxkIGBsZQ
// SIG // 7pUdU7fRZn08//+6YdI5LiIkot97zfNyzzO2A80zyEvy
// SIG // XW2OHt9VUP0JoYICKDCCAiQGCSqGSIb3DQEJBjGCAhUw
// SIG // ggIRAgEBMIGOMHcxCzAJBgNVBAYTAlVTMRMwEQYDVQQI
// SIG // EwpXYXNoaW5ndG9uMRAwDgYDVQQHEwdSZWRtb25kMR4w
// SIG // HAYDVQQKExVNaWNyb3NvZnQgQ29ycG9yYXRpb24xITAf
// SIG // BgNVBAMTGE1pY3Jvc29mdCBUaW1lLVN0YW1wIFBDQQIT
// SIG // MwAAAHGzLoprgqofTgAAAAAAcTAJBgUrDgMCGgUAoF0w
// SIG // GAYJKoZIhvcNAQkDMQsGCSqGSIb3DQEHATAcBgkqhkiG
// SIG // 9w0BCQUxDxcNMTUwNjI5MDQzNDI4WjAjBgkqhkiG9w0B
// SIG // CQQxFgQUAN9qyxjgrIBwsZM1QzhseeBLOS8wDQYJKoZI
// SIG // hvcNAQEFBQAEggEA1oRW0k431nCrlhk6xN3JVqK9D8D4
// SIG // XkuJ2gw8LYgS4voxdP4a0CrGEnzW2b3IFKFSrRN17Cm5
// SIG // dsj2GQgDjUlocEJpRRpRAqqg6NE+f/zEUO0DY6xIFCZp
// SIG // syNm7o1NnxdZwnINdncqlLPbnIqGhK73ug8WFN4Df+9Z
// SIG // zgKwXPsMe9GJAyJVzBTDJuzLmZ7HeFoOaTnBFmJaAqPP
// SIG // 1oTb29jc8cKHLR1czGlXrm1vo4b4cfuTpC2FE8cxRXCx
// SIG // Y3NNGXQ9TKT9WYafckZ2xaQLwtne82NLZ9nQJZNS6cfO
// SIG // UkWG+1qBd4HRNh1R/y3lopMsk2mJg8xmSPLdwkpju9Sw
// SIG // RKnJjQ==
// SIG // End signature block
